#include <maya/MPxFileTranslator.h>
#include <maya/MStringArray.h>
#include <maya/MSelectionList.h>
#include <maya/MItSelectionList.h>
#include <maya/MItDag.h>
#include <maya/MFnDagNode.h>
#include <maya/MFnNurbsCurve.h>
#include <maya/MPoint.h>
#include <maya/MFnTransform.h>
#include <maya/MGlobal.h>
#include <maya/MDagPath.h>
#include <maya/MFnDagNode.h>
#include <maya/MItDag.h>
#include <maya/MFnSkinCluster.h>

#include <rpspline.h>

#include "rwcommon.h"
#include "querymel.h"
#include "mayaspl.h"
#include "global.h"

RpSpline * splTranslator::ProcessNurbsCurve(MDagPath curvePath)
{
    int         numCVs;
    int         curveForm;
    int         cv;
    RwV3d       *splineVerts;
    RpSpline    *spline;
    MStatus     status = MStatus::kSuccess;

    MFnNurbsCurve fnNurbsCurve(curvePath);
    
    /* Get the number of curve control points */
    numCVs = fnNurbsCurve.numCVs(&status);
    if (status != MStatus::kSuccess)
    {
        printf("Unable to get number of curve control points\n");
        return NULL;
    }

    /* Allocate an array of spline positions */
    splineVerts = (RwV3d *)malloc(sizeof(RwV3d) * numCVs);
    if (splineVerts == NULL)
    {
        printf("Unable to allocate spline vertex data\n");
        return NULL;
    }

    /* Copy the curve control verts into the RenderWare spline verts */
    for (cv = 0; cv < numCVs; cv++)
    {
        MPoint pt;

        if (fnNurbsCurve.getCV(cv, pt, MSpace::kWorld) != MStatus::kSuccess)
        {
            printf("Unable to get curve control vert\n");
            free (splineVerts);
            return NULL;
        }
        splineVerts[cv].x = (RwReal)pt.x;
        splineVerts[cv].y = (RwReal)pt.y;
        splineVerts[cv].z = (RwReal)pt.z;

        RwV3dScale(&splineVerts[cv], &splineVerts[cv], globalData->m_scaleFactor);
    }

    /* Find if the curve is closed */
    curveForm = fnNurbsCurve.form(&status);
    if (status != MStatus::kSuccess)
    {
        printf("Unable to get curve form\n");
        free (splineVerts);
        return NULL;
    }
    
    /* Create a RenderWare spline */
    if (curveForm == MFnNurbsCurve::kOpen)
    {
        spline = RpSplineCreate (numCVs, rpSPLINETYPEOPENLOOPBSPLINE, splineVerts);
    }
    else
    {
        spline = RpSplineCreate (numCVs, rpSPLINETYPECLOSEDLOOPBSPLINE, splineVerts);
    }

    if (spline == NULL)
    {
        printf("Unable to create RenderWare spline\n");
        free (splineVerts);
        return NULL;
    }
    else
    {
        printf("Created %s spline with %d control points\n", (curveForm == MFnNurbsCurve::kOpen) ? "open" : "close" ,numCVs);
    }

    free (splineVerts);
    return spline;
}


/*
    main function for the plugin
 */
MStatus 
splTranslator::exportspl(char *fileName)
{
    MStatus         status = MStatus::kSuccess;
    RpSpline        *spline = NULL;
    RwStream        *splineStream;
    MSelectionList  selectionList;

    OpenRenderWare(NULL, false, false, false, false, false, TRUE, false, false);

    /* Get the current object selection list */
    MGlobal::getActiveSelectionList(selectionList);
    
    MItSelectionList iter(selectionList);

    /* If the selection is empty then complain */
    if (iter.isDone()) {
        printf("Nothing selected\n");
        status = MStatus::kFailure;
    }

    /* Iterate through the selection set */
    for ( ; !iter.isDone(); iter.next())
    {
        MDagPath dagPath;
        MObject component;

        status = iter.getDagPath(dagPath, component);

        /* Export the first curve we find */
        if (dagPath.hasFn(MFn::kNurbsCurve))
        {
            spline = ProcessNurbsCurve(dagPath);
            break;
        }
    }
    
    if (spline != NULL)
    {
        /* Open a file stream */
        splineStream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMWRITE, (char *)fileName);

        if (splineStream == NULL)
        {
            printf("Failed to open spline file stream.\n");
            status = MStatus::kFailure;
        }
        else
        {
            /* Write the spline */
            RpSplineStreamWrite (spline, splineStream);
            RwStreamClose(splineStream, NULL);
        }

        RpSplineDestroy(spline);
    }

    CloseRenderWare();

    printf("Export completed %s\n", status == MStatus::kSuccess ? "" : "with errors");

    /* return from the plugin command */
    return status;
}
